/************************************************************************************//**
 * @file		usbIoCard.h
 * @brief		Header file for usbioCard Class
 * @date		2013-05-08
 * @author		Research & Engineering Center Sp. z o.o.
 * @copyright	Hatteland Display AS
 ***************************************************************************************/

#ifndef USB_IO_CARD_H
#define USB_IO_CARD_H

#include "ftd2xx.h"

namespace dio
{
	/** @brief USB IO card error codes */
	enum errorCode_e
	{
		// common error codes
		NO_ERR, /**< No error */
		NA_ERR, /**< USB IO card is not connected */
		OPEN_ERR, /**< USB IO card cannot be opened */
		CLOSE_ERR, /**< USB IO card cannot be closed */
		CONFIG_ERR, /**< USB IO card has not been configured */
		OUTPUT_NAME_ERR, /**< Unsupported output name */
		OUTPUT_STATE_ERR, /**< Unsupported output state */
		OUTPUT_ERR, /**< State of an output has not been changed */
		INPUT_NAME_ERR, /**< Unsupported input name */
		INPUT_ERR, /**< State of an input has not been read */
		// internal error codes
		MPSSE_ERR, /**< USB IO card is not in MPSSE mode */
		WRITE_ERR, /**< Data cannot be written to USB IO card */
		READ_ERR, /**< Data cannot be read from USB IO card */
		NO_DATA_ERR, /**< There is no data to read in USB IO card */
		NOT_ENOUGH_DATA_ERR, /**< There is not enoug data to read in USB IO card */
	};

	/** @brief USB IO card output names */
	enum outputName_e
	{
		GPO0, /**< GPO0 output name */
		GPO1, /**< GPO1 output name */
		GPO2, /**< GPO2 output name */
		GPO3, /**< GPO3 output name */
		GPIO_NA_NAME, /**< Do not use */
	};

	/** @brief USB IO card output state names */
	enum outputStateName_e
	{
		OUT_HI, /**< Name of output high state */
		OUT_LO, /**< Name of output low state */
		OUT_NA_STATE, /**< Do not use */
	};

	/** @brief USB IO card input names */
	enum inputName_e
	{
		IN0, /**< IN0 input name */
		IN1, /**< IN1 input name */
		IN2, /**< IN2 input name */
		IN3, /**< IN3 input name */
		IN_NA_NAME, /**< Do not use */
	};

	/** @brief USB IO card input state names */
	enum inputStateName_e
	{
		IN_LO, /**< Name of input low state */
		IN_HI, /**< Name of input high state */
		IN_NA_STATE, /**< Do not use */
	};

	/** @brief A class for communicationg with USB IO card */
	class usbIoCard_c
	{
		public:
			usbIoCard_c();
			~usbIoCard_c();
			errorCode_e open();
			errorCode_e close();
			errorCode_e configure();
			errorCode_e setOutputState(outputName_e output, outputStateName_e state);
			errorCode_e diagOutputState(outputName_e output, outputStateName_e& state);
			errorCode_e readInputState(inputName_e input, inputStateName_e& state);
		private:
			FT_HANDLE ftHandle; /**< A handle to FT232 device */

			errorCode_e verifyMpsse();
			errorCode_e configIoPinDir();
			errorCode_e readData(unsigned char* dataBuffer, unsigned int bytesToRead);
			errorCode_e writeData(unsigned char* dataBuffer, unsigned int bytesToWrite);
	};
}

#endif /* USB_IO_CARD_H */
